% ---------------------------------------------------------------------

% Article:      Pricing Chinese rain: a multisite multi-period equilibrium pricing model for rainfall derivatives

% ---------------------------------------------------------------------

% Quantlet:     HaeOsiRain

% ---------------------------------------------------------------------

% Description:  Illustrates Pricing rainfall derivatives on the data of
%               China

% ---------------------------------------------------------------------

% Usage:        calles functions testmc.m, mixe.m, corrmc.m, ocsim.m, rsim.m, rsimtscor.m, rsimscorx.m, 

% ---------------------------------------------------------------------

% Inputs:       raindata.mat

% ---------------------------------------------------------------------

% Output:       Plot and tables in the paper             

% ---------------------------------------------------------------------

% Example:      -

% ---------------------------------------------------------------------

% Author:       Maria Osipenko

% ---------------------------------------------------------------------

clear

load raindata.mat
%rain occurrences
st=zeros(size(rainmm));
dt=st(:,1:2);
st(rainmm>0)=1;
dt(rainmm(:,1:2)>5)=1;
rw=14; %rolling window of size 29 centerred on the day 1:365.
ns=size(rainmm,2);%number of stations

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                     BIC Test of the order of MC (Katz,1981)
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
maxorder=4; %max order to test
s=2; %number of states
month=5; %month to test
[~,BIC]=testmc(0, maxorder, s, time, month, st);
fprintf('\nOrder of MC test (Katz,1981)\n\n');
fprintf('Order/BIC Changde   Enshi   Yichang\n');
fprintf('0          '); disp(num2str(BIC(1,:),4));
fprintf('\n1        '); disp(num2str(BIC(2,:),4));
fprintf('\n2        '); disp(num2str(BIC(3,:),4));
fprintf('\n3        '); disp(num2str(BIC(4,:),4));

month2=8; %month to test
[~,BIC]=testmc(0, maxorder, s, time, month2, dt);
fprintf('\nOrder of MC test (Katz,1981)\n\n');
fprintf('Order/BIC Changde   Enshi\n');
fprintf('0          '); disp(num2str(BIC(1,:),4));
fprintf('\n1        '); disp(num2str(BIC(2,:),4));
fprintf('\n2        '); disp(num2str(BIC(3,:),4));
fprintf('\n3        '); disp(num2str(BIC(4,:),4));
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%            Parameters of MC and rainfall amount distribution
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
[P,~,~]=testmc(1, 1, s, time, month, st); %matrix of estimated transitional probabilities
[param cumaver]=mixe(ns, time, month,rainmm); %parameters of the MoE distribution

fprintf('\nTransitional probabilities of MC and parameters of MoE distribution for RX in May\n\n');
fprintf('parameter   Changde   Enshi   Yichang\n');
fprintf('p01          '); disp(num2str(P(1,2,:),4));
fprintf('p11          '); disp(num2str(P(2,2,:),4));
fprintf('p            '); disp(num2str(param(:,1)',4));
fprintf('m1            '); disp(num2str(param(:,2)',4));
fprintf('m2           '); disp(num2str(param(:,3)',4));

[P2,~,~]=testmc(1, 1, s, time, month2, dt); %matrix of estimated transitional probabilities

fprintf('\nTransitional probabilities of MC for WX in August\n\n');
fprintf('parameter   Changde   Enshi\n');
fprintf('p01          '); disp(num2str(P2(1,2,:),4));
fprintf('p11          '); disp(num2str(P2(2,2,:),4));



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%            Empirical correlations
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

[corrmatmc, corrmatr, N]=corrmc(ns, time, month,rainmm,st);
alpha=0.05; %confidence level for CI
corrmatmclow=corrmatmc-tinv(1-alpha,N-2)*sqrt((1-corrmatmc.^2)/(N-2));
corrmatmcup=corrmatmc+tinv(1-alpha,N-2)*sqrt((1-corrmatmc.^2)/(N-2));
corrmatrlow=corrmatr-tinv(1-alpha,N-2)*sqrt((1-corrmatr.^2)/(N-2));
corrmatrup=corrmatr+tinv(1-alpha,N-2)*sqrt((1-corrmatr.^2)/(N-2));

fprintf('\nEmpirical correlations of the rainfall occurrences in May\n\n');
fprintf('parameter  value     low        high  (CI alpha 0.05) \n');
fprintf('rho12   '); disp(num2str([corrmatmc(1,2) corrmatmclow(1,2) corrmatmcup(1,2)]));
fprintf('rho13   '); disp(num2str([corrmatmc(1,3) corrmatmclow(1,3) corrmatmcup(1,3)]));
fprintf('rho23   ');disp(num2str([corrmatmc(2,3) corrmatmclow(2,3) corrmatmcup(2,3)]));

fprintf('\nEmpirical correlations of the daily rainfall in May\n\n');
fprintf('parameter  value     low        high  (CI alpha 0.05) \n');
fprintf('rho12   '); disp(num2str([corrmatr(1,2) corrmatrlow(1,2) corrmatrup(1,2)]));
fprintf('rho13   '); disp(num2str([corrmatr(1,3) corrmatrlow(1,3) corrmatrup(1,3)]));
fprintf('rho23   ');disp(num2str([corrmatr(2,3) corrmatrlow(2,3) corrmatrup(2,3)]));

[corrmatmc2, ~, N]=corrmc(ns-1, time, month2,rainmm,dt);
alpha=0.05; %confidence level for CI
corrmatmclow2=corrmatmc2-tinv(1-alpha,N-2)*sqrt((1-corrmatmc2.^2)/(N-2));
corrmatmcup2=corrmatmc2+tinv(1-alpha,N-2)*sqrt((1-corrmatmc2.^2)/(N-2));

fprintf('\nEmpirical correlations of the heavy rainfall occurrences in August\n\n');
fprintf('parameter  value     low        high  (CI alpha 0.05) \n');
fprintf('rho12   '); disp(num2str([corrmatmc2(1,2) corrmatmclow2(1,2) corrmatmcup2(1,2)]));

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%            Matching correlations of u_t's (w_ts) and v_t's
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
rhohat=corrmatmc(1,2);
N=100000; 
F=@(rho) ocsim(rho, P, N)-rhohat; %match the resulting correlation of mc to the empirical one
rho0=0.6; %start value
options=optimset('MaxIter',100000, 'MaxFunEvals',100000);
%matlabpool open local 2
tic;
rho1=fzero(F,rho0,options);

toc;
fprintf('\nCorrelation matrix of w_t for May\n\n');
[1 rho1 0;
  rho1 1 0;
  0 0 1]
rhohat2=corrmatmc2(1,2);
F=@(rho) ocsim(rho, P2, N)-rhohat2; %match the resulting correlation of mc to the empirical one
rho0=0.3; %start value
tic;
rho2=fzero(F,rho0,options);

toc;
fprintf('\nCorrelation matrix of w_t for August\n\n');
[1 rho2;
  rho2 1]
tic;
rhorhat=corrmatr(1,2);
FF=@(rhor) rsim(rho1,rhor, P, N,param)-rhorhat;
rhor0=0.3; %start value
rhor1=fzero(FF,rhor0,options);

%matlabpool close 
toc;
fprintf('\nCorrelation matrix of v_t for May\n\n');
[1 rhor1 0;
  rhor1 1 0;
  0 0 1]




matlabpool open local 5
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                          one-period-model
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%%%%%%%%%%% Parameters of the simulation%%%%%%%%%%%%%%%%%%%%%%%%

N=10000; %number of simulation steps
T=31; %days of the month to simulate

a=[0.01; 0.01];%risk aversion of farmers
am=0.01;%risk aversion of investor
r=0.05/365*T; %risk free rate in China set to 5% p.a.
R=1+r;
states=zeros(1,3); %assume that prior to start of the pricing period there was no rain in all sites
corrmatI=[0.5 0.5 0; 0.5 0 0.5];% correlation of the  farmers income with the stations 
mu=[500 500]; % mean of the incomes+
mun=[100 100]; % mean of the incomes0
mum=[50,50]; % mean of the incomes-
sigma=[100 100];%standard deviation of the incomes+
sigman=[100 100];%standard deviation of the incomes0
sigmam=[100 100];%standard deviation of the incomes-
crit=5; % critical number of wet days
K=[100 100 100]; % strikes for puts
tic;
%simulate cumulative rainfall for May and the corresponding income of farmers
[Rc0, Rc, I]=rsimtscor([0 0 0],states,N,T,rho1,rhor1,corrmatr, corrmatI, mu,sigma,P,param);
toc;
tic;
[Wx,Ix]=rsimtscorx([0 0],N,T,rho2, mu, mun, mum,sigma, sigman,sigmam,crit,P2);
toc; 
tic;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                          cumulative rainfall
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
fs1  =@(z) 1/am*1/R*1/(z(1)+z(2))*log((1/N*(ones(1,N))*exp(am*(Rc(:,1)*(z(1)+z(2))+Rc(:,2)*z(3)+Rc(:,3)*z(4))))/(1/N*(ones(1,N))*exp(am*(Rc(:,3)*z(4)+Rc(:,2)*z(3))))); %demand of investor in T-1 for station 1
fs2  =@(z) 1/am*1/R*1/(z(3))*log((1/N*(ones(1,N))*exp(am*(Rc(:,1)*(z(1)+z(2))+Rc(:,2)*z(3)+Rc(:,3)*z(4))))/(1/N*(ones(1,N))*exp(am*(Rc(:,1)*(z(1)+z(2))+Rc(:,3)*z(4))))); %demand of investor in T-1 for station 2
fs3  =@(z) 1/am*1/R*1/z(4)*log((1/N*(ones(1,N))*exp(am*(Rc(:,1)*(z(1)+z(2))+Rc(:,2)*z(3)+Rc(:,3)*z(4))))/(1/N*(ones(1,N))*exp(am*(Rc(:,1)*(z(1)+z(2))+Rc(:,2)*z(3))))); %demand of investor in T-1 for station 3

F =@(z) [1/a(1)*1/R*1/z(1)*log(1/N*(ones(1,N)*exp(-a(1)*(I(:,1)+z(3)*Rc0(:,2))))/(1/N*(ones(1,N)*exp(-a(1)*(I(:,1)+z(1)*Rc0(:,1)+z(3)*Rc0(:,2))))))-1/am*1/R*1/(z(1)+z(2))*log((1/N*(ones(1,N))*exp(am*(Rc(:,1)*(z(1)+z(2))+Rc(:,2)*z(3)+Rc(:,3)*z(4))))/(1/N*(ones(1,N))*exp(am*(Rc(:,3)*z(4)+Rc(:,2)*z(3))))); %demand of farmer 1 and supply of the investor in T-1 for W_1
1/a(1)*1/R*1/z(3)*log(1/N*(ones(1,N)*exp(-a(1)*(I(:,1)+z(1)*Rc0(:,1))))/(1/N*(ones(1,N)*exp(-a(1)*(I(:,1)+z(1)*Rc0(:,1)+z(3)*Rc0(:,2))))))-1/am*1/R*1/(z(3))*log((1/N*(ones(1,N))*exp(am*(Rc(:,1)*(z(1)+z(2))+Rc(:,2)*z(3)+Rc(:,3)*z(4))))/(1/N*(ones(1,N))*exp(am*(Rc(:,1)*(z(1)+z(2))+Rc(:,3)*z(4))))); %demand of farmer 2 and supply of the investor in T-1 for W_1
1/a(2)*1/R*1/z(2)*log(1/N*(ones(1,N)*exp(-a(2)*(I(:,2)+z(4)*Rc(:,3))))/(1/N*(ones(1,N)*exp(-a(2)*(I(:,2)+z(2)*Rc(:,1)+z(4)*Rc(:,3))))))-1/am*1/R*1/(z(1)+z(2))*log((1/N*(ones(1,N))*exp(am*(Rc(:,1)*(z(1)+z(2))+Rc(:,2)*z(3)+Rc(:,3)*z(4))))/(1/N*(ones(1,N))*exp(am*(Rc(:,3)*z(4)+Rc(:,2)*z(3))))); %demand of farmer 1 and supply of the investor in T-1 for W_2
1/a(2)*1/R*1/z(4)*log(1/N*(ones(1,N)*exp(-a(2)*(I(:,2)+z(2)*Rc(:,1))))/(1/N*(ones(1,N)*exp(-a(2)*(I(:,2)+z(2)*Rc(:,1)+z(4)*Rc(:,3))))))-1/am*1/R*1/z(4)*log((1/N*(ones(1,N))*exp(am*(Rc(:,1)*(z(1)+z(2))+Rc(:,2)*z(3)+Rc(:,3)*z(4))))/(1/N*(ones(1,N))*exp(am*(Rc(:,1)*(z(1)+z(2))+Rc(:,2)*z(3)))))]; %demand of farmer 2 and supply of the investor in T-1 for W_3

F_1s =@(z) [1/a(1)*1/R*1/z(1)*log(1/N*(ones(1,N)*exp(-a(1)*(I(:,1))))/(1/N*(ones(1,N)*exp(-a(1)*(I(:,1)+z(1)*Rc0(:,1))))))-1/am*1/R*1/(z(1)+z(2))*log((1/N*(ones(1,N))*exp(am*(Rc(:,1)*(z(1)+z(2)))))); %demand of farmer 1 and supply of the investor in T-1 for W_1
1/a(2)*1/R*1/z(2)*log(1/N*(ones(1,N)*exp(-a(2)*(I(:,2))))/(1/N*(ones(1,N)*exp(-a(2)*(I(:,2)+z(2)*Rc(:,1))))))-1/am*1/R*1/(z(1)+z(2))*log((1/N*(ones(1,N))*exp(am*(Rc(:,1)*(z(1)+z(2))))))]; %demand of farmer 2 and supply of the investor in T-1 for W_2
w=4; %size z
z0=ones(w,1);%start value for the optimization routine
options=optimset('MaxIter',100000, 'MaxFunEvals',100000);
z1=fsolve(F,z0,options); %corespond to alphas: z(1)- amount of W_1 by farmer 1 ;z(2)- amount of W_1 by farmer 2; z(3)-amount of W_2 by farmer 1, z(4)-amount of W_3 by farmer 2
W1=[fs1(z1) fs2(z1) fs3(z1)]; %correspond to the prices of cumulative may rainfall bonds
I1=I(:,1)+z1(1)*Rc0(:,1)+z1(3)*Rc0(:,2)-R*(z1(1)*W1(1)+z1(3)*W1(2));%hedged income of farmer 1
I2=I(:,2)+z1(2)*Rc(:,1)+z1(4)*Rc(:,3)-R*(z1(2)*W1(1)+z1(4)*W1(3));%hedged income of farmer 2


%single site case (Enshi)
fs1_1s  =@(z) 1/am*1/R*1/(z(1)+z(2))*log((1/N*(ones(1,N)*exp(am*(Rc(:,1)*(z(1)+z(2))))))); %demand of investor in T-1
%price for Enshi and incomes
z0_1s=[1 1];
z1_1s=fsolve(F_1s,z0_1s,options); %z(1)- F1 W1;z(2)- F2 W1; z(3)-F1 W2, z(4)-F2 W2, z(5)-F2 W3
W1_1s=fs1_1s(z1_1s);
I1_1s=I(:,1)+z1_1s(1)*Rc0(:,1)-R*z1_1s(1)*W1_1s(1);
I2_1s=I(:,2)+z1_1s(2)*Rc(:,1)-R*z1_1s(2)*W1_1s(1);

names=['no WD          '; 'WD at one site '; 'WD at two sites'];
fig=figure(1);
boxplot([I(:,1) I1_1s I1],names);hold off;
saveas(fig,'f11per.fig')
fig=figure(2);
boxplot([I(:,2) I2_1s I2],names);hold off;
saveas(fig,'f21per.fig')
toc
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                            wet days index
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
Dx=Wx;
fs1x  =@(z) 1/am*1/R*1/(z(1))*log((1/N*(ones(1,N)*exp(am*(Dx(:,1)*z(1)+Dx(:,2)*z(2)))))/(1/N*(ones(1,N)*exp(am*(Dx(:,2)*z(2)))))); %demand of investor in T-1 for station 1
fs2x  =@(z) 1/am*1/R*1/(z(2))*log((1/N*(ones(1,N)*exp(am*(Dx(:,1)*z(1)+Dx(:,2)*z(2)))))/(1/N*(ones(1,N)*exp(am*(Dx(:,1)*z(1)))))); %demand of investor in T-1 for station 2

Fx =@(z) [1/a(1)*1/R*1/z(1)*log(1/N*(ones(1,N)*exp(-a(1)*(Ix(:,1)+z(2)*Dx(:,2))))/(1/N*(ones(1,N)*exp(-a(1)*(Ix(:,1)+z(1)*Dx(:,1)+z(2)*Dx(:,2))))))-1/am*1/R*1/(z(1))*log((1/N*(ones(1,N)*exp(am*(Dx(:,1)*z(1)+Dx(:,2)*z(2)))))/(1/N*(ones(1,N)*exp(am*(Dx(:,2)*z(2)))))); %demand of farmer 1 and supply of the investor in T-1 for W_1
1/a(1)*1/R*1/z(2)*log(1/N*(ones(1,N)*exp(-a(1)*(Ix(:,1)+z(1)*Dx(:,1))))/(1/N*(ones(1,N)*exp(-a(1)*(Ix(:,1)+z(1)*Dx(:,1)+z(2)*Dx(:,2))))))-1/am*1/R*1/(z(2))*log((1/N*(ones(1,N)*exp(am*(Dx(:,1)*z(1)+Dx(:,2)*z(2)))))/(1/N*(ones(1,N)*exp(am*(Dx(:,1)*z(1))))))]; %demand of farmer 2 and supply of the investor in T-1 for W_3

F_1sx =@(z) 1/a(1)*1/R*1/z(1)*log(1/N*(ones(1,N)*exp(-a(1)*(Ix(:,1))))/(1/N*(ones(1,N)*exp(-a(1)*(Ix(:,1)+z(1)*Dx(:,1))))))-1/am*1/R*1/(z(1))*log((1/N*(ones(1,N)*exp(am*(Dx(:,1)*(z(1))))))); %demand of farmer 2 and supply of the Ixnvestor Ixn T-1 for W_2

z1x=fsolve(Fx,[1 1],options); %corespond to alphas: z(1)- amount of W_1 by farmer 1 ;z(2)- amount of W_1 by farmer 2; z(3)-amount of W_2 by farmer 1, z(4)-amount of W_3 by farmer 2
W1x=[fs1x(z1x) fs2x(z1x)]; %correspond to the prices of cumulative may rainfall bonds

%single site case (Enshi)
fs1_1sx  =@(z) 1/am*1/R*1/(z(1))*log((1/N*(ones(1,N)*exp(am*(Dx(:,1)*(z(1))))))); %demand of investor in T-1
%price for Enshi and incomes
z1_1sx=fsolve(F_1sx,1,options); %z(1)- F1 W1;z(2)- F2 W1; z(3)-F1 W2, z(4)-F2 W2, z(5)-F2 W3
W1_1sx=fs1_1sx(z1_1sx);


%demand and supply curves for one and two sites
%investor
fs1_1sz  =@(z11) 1/am*1/R*1/(z11)*log(1/N*(ones(1,N)*exp(am*(Dx(:,1)*(z11))))); %supply of investor in T-1 with only Enshi traded
fs1z  =@(z11) 1/am*1/R*1/(z11)*log((1/N*(ones(1,N)*exp(am*(Dx(:,1)*(z11)+Dx(:,2)*z1x(2)))))/(1/N*(ones(1,N)*exp(am*(Dx(:,2)*z1x(2)))))); %supply of investor in T-1 with all stations traded
%farmer 1
fb1 =@(z11) 1/a(1)*1/R*1/z11*log(1/N*(ones(1,N)*exp(-a(1)*(Ix(:,1)+z1x(2)*Dx(:,2))))/(1/N*(ones(1,N)*exp(-a(1)*(Ix(:,1)+z1x(2)*Dx(:,2)+z11*Dx(:,1)))))); %demand of the farmer 2 for W_3
fb1_1s  =@(z11) 1/a(1)*1/R*1/z11*log(1/N*(ones(1,N)*exp(-a(1)*(Ix(:,1))))/(1/N*(ones(1,N)*exp(-a(1)*(Ix(:,1)+z11*Dx(:,1)))))); % demand of farmer 2 for W_3 in multi-site case


zz=(0.1:0.5:50)'; 
%zz=(0.1:5:1000)';
covt=NaN(length(zz),2);
fzz1=NaN(length(zz),2); fzz11=NaN(length(zz),2); fzz2=NaN(length(zz),2); fzz22=NaN(length(zz),2);  fzz3=NaN(length(zz),2); fzz33=NaN(length(zz),2);
for i=1:length(zz)
%theorem 2.4
s1_1=exp(-a(1)*Ix(:,1)); s2_1=exp(-a(1)*zz(i)*Dx(:,1)); s3_1=exp(-a(1)*z1x(2)*Dx(:,2));
cov23_1=cov(s2_1,s3_1); cov12_1=cov(s1_1,s2_1);  cov13_1=cov(s1_1,s3_1);
covt(i,1)=cov23_1(1,2)*(mean(s1_1))^2-cov12_1(1,2)*cov13_1(1,2)+mean(s1_1)*mean((s1_1-mean(s1_1)).*(s2_1-mean(s2_1)).*(s3_1-mean(s3_1)));

fzz1(i,:)=fs1_1sz(zz(i)); fzz11(i,:)=fs1z(zz(i)); 
fzz2(i,:)=fb1(zz(i)); fzz22(i,:)=fb1_1s(zz(i)); 
end
[min(zz(covt(:,1)>0)) max(zz(covt(:,1)>0))]


fig=figure(3);
plot(zz',fzz2,'Color',[0 0 1]); hold on; plot(zz',fzz22,'Color',[1 0 0]);
plot(z1x(1),W1x(1),'o'); plot(z1_1sx(1),W1_1sx(1),'o');
text(z1x(1),W1x(1)-0.2,'Multi-site equilibrium'); text(z1_1sx(1)+3,W1_1sx(1),'Single site equilibrium');
hold off; 
saveas(fig, 'Sfar1.fig')

fig=figure(4);
plot(zz',fzz2,'Color',[0 0 1]); hold on; plot(zz',fzz22,'Color',[1 0 0]);
plot(z1x(1),W1x(1),'o'); plot(z1_1sx(1),W1_1sx(1),'o');
plot(zz', fzz1,'Color',[1 0 0]); plot(zz',fzz11,'Color',[0 0 1]);
text(z1x(1),W1x(1)+0.2,'Multi-site equilibrium'); text(z1_1sx(1),W1_1sx(1)-0.2,'Single site equilibrium');
hold off; 
saveas(fig, 'demsup1pern.fig')

fig=figure(5);
plot(zz', fzz1,'Color',[1 0 0]); hold on; plot(zz',fzz11,'Color',[0 0 1]);
plot(z1x(1),W1x(1),'o'); plot(z1_1sx(1),W1_1sx(1),'o');
text(z1x(1),W1x(1)+0.2,'Multi-site equilibrium'); text(z1_1sx(1)+3,W1_1sx(1),'Single site equilibrium');
hold off; 
saveas(fig, 'Dinv.fig')


I1x=Ix+z1x(1)*Wx(:,1)+z1x(2)*Wx(:,2)-R*(z1x(1)*W1x(1)+z1(2)*W1x(2));%hedged income of farmer 1
I1_1sx=Ix+z1_1sx(1)*Wx(:,1)-R*z1_1sx(1)*W1_1sx(1);

fig=figure(5);
hr=boxplot([Ix I1_1sx I1x],names);
for ih=1:3
set(hr(ih,:),'LineWidth',1);
end
saveas(fig,'f11perx.fig')

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%Different strikes
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
Kcrit=[1 2 3 4 5 6 7];
Wc=NaN(2,length(Kcrit));
for o=1:length(Kcrit)
    c=Kcrit(o);
    [Dx,Ix]=rsimtscorx([0 0],N,T,rho2, mu, mun, mum,sigma, sigman,sigmam,c,P2);
fs1x  =@(z) 1/am*1/R*1/(z(1))*log((1/N*(ones(1,N)*exp(am*(Dx(:,1)*z(1)+Dx(:,2)*z(2)))))/(1/N*(ones(1,N)*exp(am*(Dx(:,2)*z(2)))))); %demand of investor in T-1 for station 1
fs2x  =@(z) 1/am*1/R*1/(z(2))*log((1/N*(ones(1,N)*exp(am*(Dx(:,1)*z(1)+Dx(:,2)*z(2)))))/(1/N*(ones(1,N)*exp(am*(Dx(:,1)*z(1)))))); %demand of investor in T-1 for station 2

Fx =@(z) [1/a(1)*1/R*1/z(1)*log(1/N*(ones(1,N)*exp(-a(1)*(Ix(:,1)+z(2)*Dx(:,2))))/(1/N*(ones(1,N)*exp(-a(1)*(Ix(:,1)+z(1)*Dx(:,1)+z(2)*Dx(:,2))))))-1/am*1/R*1/(z(1))*log((1/N*(ones(1,N)*exp(am*(Dx(:,1)*z(1)+Dx(:,2)*z(2)))))/(1/N*(ones(1,N)*exp(am*(Dx(:,2)*z(2)))))); %demand of farmer 1 and supply of the investor in T-1 for W_1
1/a(1)*1/R*1/z(2)*log(1/N*(ones(1,N)*exp(-a(1)*(Ix(:,1)+z(1)*Dx(:,1))))/(1/N*(ones(1,N)*exp(-a(1)*(Ix(:,1)+z(1)*Dx(:,1)+z(2)*Dx(:,2))))))-1/am*1/R*1/(z(2))*log((1/N*(ones(1,N)*exp(am*(Dx(:,1)*z(1)+Dx(:,2)*z(2)))))/(1/N*(ones(1,N)*exp(am*(Dx(:,1)*z(1))))))]; %demand of farmer 2 and supply of the investor in T-1 for W_3

F_1sx =@(z) 1/a(1)*1/R*1/z(1)*log(1/N*(ones(1,N)*exp(-a(1)*(Ix(:,1))))/(1/N*(ones(1,N)*exp(-a(1)*(Ix(:,1)+z(1)*Dx(:,1))))))-1/am*1/R*1/(z(1))*log((1/N*(ones(1,N)*exp(am*(Dx(:,1)*(z(1))))))); %demand of farmer 2 and supply of the Ixnvestor Ixn T-1 for W_2

zc=fsolve(Fx,[1 1],options); %corespond to alphas: z(1)- amount of W_1 by farmer 1 ;z(2)- amount of W_1 by farmer 2; z(3)-amount of W_2 by farmer 1, z(4)-amount of W_3 by farmer 2
Wc(:,o)=[fs1x(zc) fs2x(zc)]; %correspond to the prices of cumulative may rainfall bonds
end;

fig=figure(8);
plot(Kcrit,Wc)



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                             two-period-model
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


NN=10^4;
states=zeros(1,3);
T=16; k=3;
grid=zeros(1,3);
[Rc0,Rc, ~,sts,rpath]=rsimtscor(grid,states,NN,T,rho1,rhor1,corrmatr, corrmatI, mu,sigma,P,param);
Alpha=NaN(NN,w);
Price=NaN(NN,k);
State=NaN(NN,k);
Th=NaN(NN,2);
Thm=NaN(NN,1);
Alphaal=NaN(NN,w);
Priceal=NaN(NN,k);
Stateal=NaN(NN,k);
Thal=NaN(NN,2);
grid=Rc;
r=0.053/365*T;
R=1+r;
options=optimset('MaxIter',1000, 'MaxFunEvals',1000);

%for one-site-model
Alpha_1s=NaN(NN,2);
Price_1s=NaN(NN,1);
State_1s=NaN(NN,k);
Th_1s=NaN(NN,2);
I1_2=NaN(NN,NN);
I1_21s=NaN(NN,NN);
I2_2=NaN(NN,NN);
I2_21s=NaN(NN,NN);

tic;
for s=1:NN 
[Rc0,Rc,I]=rsimtscor(grid(s,:),sts(s,:),NN,T,rho1,rhor1, corrmatr, corrmatI, mu,sigma,P,param);

fs1  =@(z) 1/am*1/R*1/(z(1)+z(2))*log((1/NN*(ones(1,NN))*exp(am*(Rc(:,1)*(z(1)+z(2))+Rc(:,2)*z(3)+Rc(:,3)*z(4))))/(1/NN*(ones(1,NN))*exp(am*(Rc(:,3)*z(4)+Rc(:,2)*z(3))))); %demand of investor in T-1
fs2  =@(z) 1/am*1/R*1/(z(3))*log((1/NN*(ones(1,NN))*exp(am*(Rc(:,1)*(z(1)+z(2))+Rc(:,2)*z(3)+Rc(:,3)*z(4))))/(1/NN*(ones(1,NN))*exp(am*(Rc(:,1)*(z(1)+z(2))+Rc(:,3)*z(4))))); %demand of investor in T-1
fs3  =@(z) 1/am*1/R*1/z(4)*log((1/NN*(ones(1,NN))*exp(am*(Rc(:,1)*(z(1)+z(2))+Rc(:,2)*z(3)+Rc(:,3)*z(4))))/(1/NN*(ones(1,NN))*exp(am*(Rc(:,1)*(z(1)+z(2))+Rc(:,2)*z(3))))); %demand of investor in T-1

F =@(z) [1/a(1)*1/R*1/z(1)*log(1/NN*(ones(1,NN)*exp(-a(1)*(I(:,1)+z(3)*Rc0(:,2))))/(1/NN*(ones(1,NN)*exp(-a(1)*(I(:,1)+z(1)*Rc0(:,1)+z(3)*Rc0(:,2))))))-1/am*1/R*1/(z(1)+z(2))*log((1/NN*(ones(1,NN))*exp(am*(Rc(:,1)*(z(1)+z(2))+Rc(:,2)*z(3)+Rc(:,3)*z(4))))/(1/NN*(ones(1,NN))*exp(am*(Rc(:,3)*z(4)+Rc(:,2)*z(3))))); %demand of farmer in T-1 for W_1
1/a(1)*1/R*1/z(3)*log(1/NN*(ones(1,NN)*exp(-a(1)*(I(:,1)+z(1)*Rc0(:,1))))/(1/NN*(ones(1,NN)*exp(-a(1)*(I(:,1)+z(1)*Rc0(:,1)+z(3)*Rc0(:,2))))))-1/am*1/R*1/(z(3))*log((1/NN*(ones(1,NN))*exp(am*(Rc(:,1)*(z(1)+z(2))+Rc(:,2)*z(3)+Rc(:,3)*z(4))))/(1/NN*(ones(1,NN))*exp(am*(Rc(:,1)*(z(1)+z(2))+Rc(:,3)*z(4))))); %demand of farmer in T-1 for W_2
1/a(2)*1/R*1/z(2)*log(1/NN*(ones(1,NN)*exp(-a(2)*(I(:,2)+z(4)*Rc(:,3))))/(1/NN*(ones(1,NN)*exp(-a(2)*(I(:,2)+z(2)*Rc(:,1)+z(4)*Rc(:,3))))))-1/am*1/R*1/(z(1)+z(2))*log((1/NN*(ones(1,NN))*exp(am*(Rc(:,1)*(z(1)+z(2))+Rc(:,2)*z(3)+Rc(:,3)*z(4))))/(1/NN*(ones(1,NN))*exp(am*(Rc(:,3)*z(4)+Rc(:,2)*z(3))))); %demand of farmer in T-1 in st 1
1/a(2)*1/R*1/z(4)*log(1/NN*(ones(1,NN)*exp(-a(2)*(I(:,2)+z(2)*Rc(:,1))))/(1/NN*(ones(1,NN)*exp(-a(2)*(I(:,2)+z(2)*Rc(:,1)+z(4)*Rc(:,3))))))-1/am*1/R*1/z(4)*log((1/NN*(ones(1,NN))*exp(am*(Rc(:,1)*(z(1)+z(2))+Rc(:,2)*z(3)+Rc(:,3)*z(4))))/(1/NN*(ones(1,NN))*exp(am*(Rc(:,1)*(z(1)+z(2))+Rc(:,2)*z(3)))))]; %demand of farmer in T-1 in st 3

F_1s =@(z) [1/a(1)*1/R*1/z(1)*log(1/NN*(ones(1,NN)*exp(-a(1)*(I(:,1))))/(1/NN*(ones(1,NN)*exp(-a(1)*(I(:,1)+z(1)*Rc0(:,1))))))-1/am*1/R*1/(z(1)+z(2))*log((1/NN*(ones(1,NN))*exp(am*(Rc(:,1)*(z(1)+z(2)))))); %demand of farmer in T-1 for W_1
1/a(2)*1/R*1/z(2)*log(1/NN*(ones(1,NN)*exp(-a(2)*(I(:,2))))/(1/NN*(ones(1,NN)*exp(-a(2)*(I(:,2)+z(2)*Rc(:,1))))))-1/am*1/R*1/(z(1)+z(2))*log((1/NN*(ones(1,NN))*exp(am*(Rc(:,1)*(z(1)+z(2))))))]; %demand of farmer in T-1 for W_2

%single site case (Enshi)
fs1_1s  =@(z) 1/am*1/R*1/(z(1)+z(2))*log((1/NN*(ones(1,NN))*exp(am*(Rc(:,1)*(z(1)+z(2)))))); %demand of investor in T-1
a=[0.01 0.01 0.01];

Alpha0=fsolve(F,z0,options);
Alpha(s,:)=Alpha0;
Price0=[fs1(Alpha0) fs2(Alpha0) fs3(Alpha0)];
Price(s,:)=Price0;
State(s,:)=grid(s,:);
Th(s,:)=[exp(a(1)*R*(Alpha0(1)*Price0(1)+Alpha0(3)*Price0(2)))*(1/NN*(ones(1,NN)*exp(-a(1)*(I(:,1)+(Alpha0(1)*Rc0(:,1)+Alpha0(3)*Rc0(:,2)))))) exp(a(2)*R*(Alpha0(2)*Price0(1)+Alpha0(4)*Price0(3)))*(1/NN*(ones(1,NN)*exp(-a(2)*(I(:,2)+(Alpha0(2)*Rc(:,1)+Alpha0(4)*Rc(:,3))))))]; %#ok<*PFSAME>
Thm(s,:)=exp(-am*R*(Alpha0(1)*Price0(1)+Alpha0(2)*Price0(1)+Alpha0(3)*Price0(2)+Alpha0(4)*Price0(3)))*(1/NN*(ones(1,NN)*exp(am*(sum(Alpha0(1:2))*Rc(:,1)+Alpha0(3)*Rc(:,2)+Alpha0(4)*Rc(:,3)))));

%one-site case
Alpha0_1s=fsolve(F_1s,z0_1s(1:2),options);
Alpha_1s(s,:)=Alpha0_1s;
Price0_1s=fs1_1s(Alpha0_1s);
Price_1s(s,:)=Price0_1s;
Th_1s(s,:)=[exp(a(1)*R*(Alpha0(1)*Price0(1)))*(1/NN*(ones(1,NN)*exp(-a(1)*(I(:,1)+(Alpha0(1)*Rc0(:,1)))))) exp(a(2)*R*(Alpha0(2)*Price0(1)))*(1/NN*(ones(1,NN)*exp(-a(2)*(I(:,2)+(Alpha0(2)*Rc(:,1))))))]; %#ok<*PFSAME>

%income distribution in T

I1_2(:,s)=I(:,1)+(Alpha(s,1)*Rc0(:,1)+Alpha(s,3)*Rc0(:,2));
I2_2(:,s)=I(:,2)+(Alpha(s,2)*Rc(:,1)+Alpha(s,4)*Rc(:,3));

I1_21s(:,s)=I(:,1)+Alpha_1s(s,1)*Rc0(:,1);
I2_21s(:,s)=I(:,2)+Alpha_1s(s,2)*Rc(:,1);

end;
toc;

fs1_2  =@(z) 1/am*1/R^2*1/(z(1)+z(2))*log((1/NN*(ones(1,NN)*(Thm.*exp(am*R*(Price(:,1)*(z(1)+z(2))+Price(:,2)*z(3)+Price(:,3)*z(4))))))/(1/NN*(ones(1,NN)*(Thm.*exp(am*R*(Price(:,3)*z(4)+Price(:,2)*z(3))))))); %demand of investor in T-2
fs2_2  =@(z) 1/am*1/R^2*1/(z(3))*log((1/NN*(ones(1,NN)*(Thm.*exp(am*R*(Price(:,1)*(z(1)+z(2))+Price(:,2)*(z(3))+Price(:,3)*z(4))))))/(1/NN*(ones(1,NN)*(Thm.*exp(am*R*(Price(:,1)*(z(1)+z(2))+Price(:,3)*z(4))))))); %demand of investor in T-2
fs3_2  =@(z) 1/am*1/R^2*1/z(4)*log((1/NN*(ones(1,NN)*(Thm.*exp(am*R*(Price(:,1)*(z(1)+z(2))+Price(:,2)*(z(3))+Price(:,3)*z(4))))))/(1/NN*(ones(1,NN)*(Thm.*exp(am*R*(Price(:,1)*(z(1)+z(2))+Price(:,2)*(z(3)))))))); %demand of investor in T-2

F_2 =@(z) [1/a(1)*1/R^2*1/z(1)*log((1/NN*(ones(1,NN)*(Th(:,1).*exp(-a(1)*(z(3)*Price(:,2)*R)))))/(1/NN*(ones(1,NN)*(Th(:,1).*exp(-a(1)*R*(z(1)*Price(:,1)+z(3)*Price(:,2)))))))-1/am*1/R^2*1/(z(1)+z(2))*log((1/NN*(ones(1,NN)*(Thm.*(exp(am*R*(Price(:,1)*(z(1)+z(2))+Price(:,2)*z(3)+Price(:,3)*z(4)))))))/(1/NN*(ones(1,NN)*(Thm.*(exp(am*R*(Price(:,3)*z(4)+Price(:,2)*z(3)))))))); %demand of farmer - investor in T-2 for W_1
1/a(1)*1/R^2*1/z(3)*log((1/NN*(ones(1,NN)*(Th(:,1).*exp(-a(1)*(z(1)*Price(:,1)*R)))))/(1/NN*(ones(1,NN)*(Th(:,1).*exp(-a(1)*R*(z(1)*Price(:,1)+z(3)*Price(:,2)))))))-1/am*1/R^2*1/(z(3))*log((1/NN*(ones(1,NN)*(Thm.*(exp(am*R*(Price(:,1)*(z(1)+z(2))+Price(:,2)*(z(3))+Price(:,3)*z(4)))))))/(1/NN*(ones(1,NN)*(Thm.*(exp(am*R*(Price(:,1)*(z(1)+z(2))+Price(:,3)*z(4)))))))); %demand of farmer - investor in T-2 for W_2
1/a(2)*1/R^2*1/z(2)*log((1/NN*(ones(1,NN)*(Th(:,2).*exp(-a(2)*(z(4)*Price(:,3))*R))))/(1/NN*(ones(1,NN)*(Th(:,2).*exp(-a(2)*R*(z(2)*Price(:,1)+z(4)*Price(:,3)))))))-1/am*1/R^2*1/(z(1)+z(2))*log((1/NN*(ones(1,NN)*(Thm.*(exp(am*R*(Price(:,1)*(z(1)+z(2))+Price(:,2)*z(3)+Price(:,3)*z(4)))))))/(1/NN*(ones(1,NN)*(Thm.*(exp(am*R*(Price(:,3)*z(4)+Price(:,2)*z(3)))))))); %demand of farmer - investor in T-2 for W_1
1/a(2)*1/R^2*1/z(4)*log((1/NN*(ones(1,NN)*(Th(:,2).*exp(-a(2)*(z(2)*Price(:,1))*R))))/(1/NN*(ones(1,NN)*(Th(:,2).*exp(-a(2)*R*(z(2)*Price(:,1)+z(4)*Price(:,3)))))))-1/am*1/R^2*1/z(4)*log((1/NN*(ones(1,NN)*(Thm.*(exp(am*R*(Price(:,1)*(z(1)+z(2))+Price(:,2)*(z(3))+Price(:,3)*z(4)))))))/(1/NN*(ones(1,NN)*(Thm.*(exp(am*R*(Price(:,1)*(z(1)+z(2))+Price(:,2)*(z(3)))))))))]; %demand of farmer - investor in T-2 in W_3

options=optimset('MaxIter',1000000, 'MaxFunEvals',1000000, 'TolFun' , 1e-8,'TolX',1e-6);
Alpha0=fsolve(F_2,z0,options);
Price0=[fs1_2(Alpha0) fs2_2(Alpha0) fs3_2(Alpha0)];

%one-site case
fs1_21s  =@(z) 1/am*1/R^2*1/(z(1)+z(2))*log(1/NN*(ones(1,NN)*(exp(am*R*(Price(:,1)*(z(1)+z(2))))))); %demand of investor in T-2

F_21s =@(z) [1/a(1)*1/R^2*1/z(1)*log(1/NN*(ones(1,NN)*(Th(:,1)))/(1/NN*(ones(1,NN)*(Th(:,1).*exp(-a(1)*R*(z(1)*Price(:,1)))))))-1/am*1/R^2*1/(z(1)+z(2))*log(1/NN*(ones(1,NN)*(exp(am*R*(Price(:,1)*(z(1)+z(2))))))); %demand of farmer - investor in T-2 for W_1
1/a(2)*1/R^2*1/z(2)*log(1/NN*(ones(1,NN)*(Th(:,2)))/(1/NN*(ones(1,NN)*(Th(:,2).*exp(-a(2)*R*(z(2)*Price(:,1)))))))-1/am*1/R^2*1/(z(1)+z(2))*log(1/NN*(ones(1,NN)*(exp(am*R*(Price(:,1)*(z(1)+z(2)))))))]; %demand of farmer - investor in T-2 for W_2

Alpha0_1s=fsolve(F_21s,z0_1s(1:2),options);
Price0_1s=fs1_1s(Alpha0_1s);


%income distribution
for s=1:NN
I1_2(:,s)=I1_2(:,s)+Alpha0(1)*(Price(s,1)-Price0(1)*R)*R+Alpha0(3)*(Price(s,2)-Price0(2)*R)*R-Alpha(s,1)*Price(s,1)*R-Alpha(s,3)*Price(s,2)*R;
I2_2(:,s)=I2_2(:,s)+Alpha0(2)*(Price(s,1)-Price0(1)*R)*R+Alpha0(4)*(Price(s,3)-Price0(3)*R)*R-Alpha(s,2)*Price(s,1)*R-Alpha(s,4)*Price(s,3)*R;

I1_21s(:,s)=I1_21s(:,s)+Alpha0_1s(1)*(Price_1s(s,1)-Price0_1s(1)*R)*R-Alpha_1s(s,1)*Price_1s(s,1)*R;
I2_21s(:,s)=I2_21s(:,s)+Alpha0_1s(2)*(Price_1s(s,1)-Price0_1s(1)*R)*R-Alpha_1s(s,2)*Price_1s(s,1)*R;
end
save 2per_N10000.mat

I=mu(1)+sigma(1)*randn(NN^2,1);

fig=figure(5);
hl=boxplot([I reshape(I1_21s,NN^2,1) reshape(I1_2,NN^2,1)],names);
for ih=1:3
set(hl(ih,:),'LineWidth',1.2);
end
saveas(fig,'f12per.fig')
fig=figure(6);
hr=boxplot([I reshape(I2_21s,NN^2,1) reshape(I2_2,NN^2,1)],names);
for ih=1:3
set(hr(ih,:),'LineWidth',1.2);
end
saveas(fig,'f22per.fig')

fig=figure(7);

subplot(2,1,1)
plot(1:16, [rpath(1:16,1)'; rpath(1:16,2)'; rpath(1:16,3)'],'LineWidth',2); hold on;
plot(16:31, zeros(16,1), 'Color','black');
text(5,70,'Changde','Color',[0 0 1]); 
text(12,30,'Enshi','Color',[0 0.498039215803146 0]);
text(12,80,'Yichang','Color',[1 0 0]);
plot(repmat(16,length(1:250),1),1:250,'LineWidth',1,'LineStyle','--');hold off;

subplot(2,1,2)
plot(1:31, repmat(W1,31,1),'LineStyle','--','LineWidth',2); hold on;
plot(1:16,repmat(Price0,16,1),'LineWidth',2);
plot(16:31, repmat(Price(end,:),16,1),'LineWidth',2); hold off;

saveas(fig,'example.fig')


I12=reshape(I1_21s,NN^2,1);
I21=reshape(I2_21s,NN^2,1);
I1=reshape(I1_2,NN^2,1);
I2=reshape(I2_2,NN^2,1);
I=mu(1)+sigma(1)*randn(NN^2,1);


fig=figure(8);
names=[' no WD   '; '  WD at  '; 'one site '; ' WDs at  '; 'two sites'];
boxplot([I I12 I21 I1 I2],'colors','k','symbol','k+','labels',names);
%hr=boxplot([I I12 I21 I1 I2],'colors','k','labels',names);
set(gca,'FontSize',12)
for ih=1:3
set(hr(ih,:),'LineWidth',1.2);
end