proc()=VARDSFM(y,origyear,origperiod,periodicity,name)
; -----------------------------------------------------------------
;   Library       multi
; -----------------------------------------------------------------
;   See_also      MulTirr, MulTiss,  MulTici  
; -----------------------------------------------------------------
;   Macro         VARDSFM
; -----------------------------------------------------------------
;   Description   general analysis for the Full VAR Model
; -----------------------------------------------------------------
;   Definition    k x n matrix, the original time series
; -----------------------------------------------------------------
;   Usage        modelbeta(y{,origyear{,origperiod{,periodicity{,name}}}})  
;   Input
;     Parameter   y  
;     Definition      n x p matrix, data of factor loadings time series
;     Parameter   origyear 
;     Definition      integer, the year of the first observation 
;     Parameter   origperiod 
;     Definition      integer, the season of the first observation
;     Parameter   periodicity
;     Definition      integer, the number of seasons per period
;     Parameter   name
;     Definition      string vector, names of the factor loadings time series
; -----------------------------------------------------------------
;   Example       VAR-DSFM modeling
;                 library("multi")
;		  loadings=read("        ")    ; read in the "FactorLoadings-data"
;		  loadings=loadings[,3:5] 
;		  level=loadings[,1] 
;		  slope=loadings[,2] 
;		  curvature=loadings[,3] 
;		  VARDSFM(loadings,1999,1,251,"level factor"|"slope factor"|"curvature factor")
; -----------------------------------------------------------------
;   Result      Menu "Full VAR Estimation" is created;
;               it allows for interactive analysis.
; -----------------------------------------------------------------
;   Keywords    Time Series Analysis
; ---------------------------------------------------------------------
;   Author      J. Mungo 20050120
;               license MD*Tech
; -----------------------------------------------------------------

; initialisation, create header
  nocancel = 1
  k = cols(y)
  n = rows(y)
  yt = y
  type = 0
  outp = 0
  filename=""
  typestring = "full"|""|""|""
  natlog = matrix(k,1)*0
  diff = matrix(k,1)*0
  exclude = matrix(k,1)*0
  if (exist(name)<=0)
    name = string("Time series 1.0f",1:k)
  endif
  if (exist(origyear)<=0)
    origyear = 1
  endif
  if (exist(periodicity)<=0)
    periodicity = 1
  endif
  if (exist(origperiod)<=0)
    origperiod = 1
  endif
  header = list(natlog,diff,origyear,periodicity,origperiod,name,exclude)

  while (nocancel)

; multi - main menu-----------------------------

    tasknames = "mgraphic"|"preliminary"|"type"|"specification"
    tasknames = tasknames|"output"
    head = " Full VAR Estimation"
    item = " Multiple Time Series Plot"
    item = item | " Transformations & pre-analysis"
    if (type>0)
      item = item | " Model type ("+typestring[type]+")"
    else
      item = item | " Model type"
    endif
    item = item | " Model specification"; and estimation"
    if (outp == 1)
      item = item | " Output filename ("+filename+") "
    else
      item = item | " Output filename"
    endif
    item = item | " Quit Frame"
    task = selectitem(head,item,"single")
    if ((task[rows(item)])||sum(task)==0)
      nocancel = 0
      task = " "
    else 
      task = paf(tasknames,task[1:rows(item)-1])
    endif

; Multiple Graphic--------------------------------

    if ((nocancel)&&(task=="mgraphic"))
      z=yt
      n=rows(z)
      kd=cols(z)
     
      t = (1:rows(y))/251 .+ origyear
      setsize(900, 400)
      dmulti=createdisplay(1,1)
      w=t~z[,1]
      
      w=setmask(w,"border",0,"empty","line","medium","solid")
      th = aseq(1,rows(w),1)
      setmaskl(w,th',1,1,2)
      show(dmulti,1,1,w)
      mmin=min(min(z)')
      mmax=max(max(z)')
      sp=(mmax-mmin)/20
      setgopt(dmulti,1,1,"ylim",mmin-sp|mmax+sp,"title"," Time plot of factor loadings: level (blue), slope(green) and curvature(cyan)")
      i=2
      while (i<=kd)
        w=t~z[,i]	
        w=setmask(w,"border",0,"empty","line","medium","solid")  
        setmaskl(w,th',i,1,2)
        adddata(dmulti,1,1,w)
        i=i+1
      endo   
    endif

; Transformations and preliminary analysis-----------------------------

    yprelim = y
    s = paf((1:k),abs(header.exclude-1))
    yprelim[,s] = yt[,s]
    noquit = 1
    while ((noquit)&&(nocancel)&&(task=="preliminary"))
      head = " Select time series"
      item = string(" ",1:k)+header.name+string("                ",1:k)
      item = substr(item,1,16)
      logstr = string("   ",1:k)
      if (sum(header.natlog)>0)
        logstr[paf(1:k,header.natlog)] = "log"
      endif
      item = item + logstr
      diffstr = string("                    ",1:k)
      if (sum(header.diff)>0)
        sel = paf(1:k,header.diff)
        diffstr[sel] = string("  1.differences(%1.0f)  ",header.diff[sel])
      endif
      item = item + diffstr
      excludestr = string("        ",1:k)
      if (sum(header.exclude)>0)
        excludestr[paf(1:k,header.exclude)] = "excluded"
      endif
      item = item + excludestr
      item = item|" Back"
      tseries = selectitem(head,item,"single")'*(1:rows(item))
      if (tseries==0||tseries==rows(item))
        noquit = 0
        noback = 0
      else
        noback = 1
      endif
      while (noback)
        head = " Preanalysis: " + header.name[tseries]
        if (header.exclude[tseries])
          item = " Include"
        else
          item = " Exclude"
        endif
        if (header.natlog[tseries])
          item = item|" Remove natural logarithm"
        else
          item = item|" Natural logarithm"
        endif
        if (header.diff[tseries]==0)
          item = item|" 1. Differences"
        else
          item = item|" Remove 1. Differences "+string("(%1.0f)",header.diff[tseries])
        endif
        item = item|" Graphic"
        item = item|" ACF"
        item = item|" PACF"
        item = item|" Periodogram"
        item = item|" Spectral density estimation"
        item = item|" Back"
        prelim = selectitem(head,item,"single")
 
        switch
        case (prelim[1])
          if ((sum(header.exclude)<(k-1))||(header.exclude[tseries]))
            header.exclude[tseries] = abs(header.exclude[tseries]-1)
          endif
          break
        case (prelim[2])
          header.natlog[tseries] = abs(header.natlog[tseries]-1)
          if (header.natlog[tseries])
            yprelim[,tseries] = log(y[,tseries])
          else
            yprelim[,tseries] = y[,tseries]
          endif
          if (header.diff[tseries]>0)
            yprelim[,tseries] = tdiff(yprelim[,tseries],header.diff[tseries])

          endif
          break
        case (prelim[3])
          if (header.diff[tseries]==0)
            dstr = " Difference lag"
            dval = 1
            dval = readvalue(dstr,dval)
            header.diff[tseries] = dval
          else
            header.diff[tseries] = 0
          endif
          if (header.natlog[tseries])
            yprelim[,tseries] = log(y[,tseries])
            p=1
          else
            yprelim[,tseries] = y[,tseries]
            p=0
          endif
          if (header.diff[tseries]>0)
            yprelim[,tseries] = tdiff(yprelim[,tseries],header.diff[tseries])
          
          endif
          break
        case (prelim[4])
          hheader = header
          h = (header.origyear*header.periodicity)+header.origperiod
          h = h + header.diff[tseries]
          hheader.origyear = floor(h/header.periodicity)
          hheader.origperiod = h - (hheader.origyear*header.periodicity)
          if ((header.natlog[tseries])&& (header.diff[tseries]>0))
            hheader.name = "log differences " + header.name[tseries]
          else
            if (header.natlog[tseries])
              hheader.name = "log " + header.name[tseries]
            else
              if  (header.diff[tseries]>0)
                hheader.name = "1. differences " + header.name[tseries]
              else
                hheader.name = header.name[tseries]
              endif
            endif
          endif
          dummy=timeplot(yprelim[(header.diff[tseries]+1):n,tseries],400,hheader)
          break
        case (prelim[5])
          acfplot(yprelim[(header.diff[tseries]+1):n,tseries])
          if (outp==1)
            "ACF of " + header.name[tseries] 
            acf(yprelim[(header.diff[tseries]+1):n,tseries])
          endif
          break
        case (prelim[6])
          pacfplot(yprelim[(header.diff[tseries]+1):n,tseries])
          if (outp==1)
            "PACF of " + header.name[tseries] 
            pacf(yprelim[(header.diff[tseries]+1):n,tseries],30)
          endif
          break
        case (prelim[7])
          pg = pgram(yprelim[(header.diff[tseries]+1):n,tseries])
          if (outp==1)
            "Periodogram of " + header.name[tseries] 
            pg
          endif
          break
        case (prelim[8])
          sp = spec(yprelim[(header.diff[tseries]+1):n,tseries])
          if (outp==1)
            "Spectral Density Estimate of " + header.name[tseries] 
            sp
          endif
        case (prelim[9])
          noback = 0
        endsw
      endo
      dummy  = 0
      dpgram = 0
      dspec  = 0
      dpgram = 0
      yt = paf(yprelim',abs(header.exclude-1))'
    endo
 
; Model type---------------------------------

    if ((nocancel)&&(task=="type"))
      head = " Select model type"
      item =        " Full VAR Model"
      type = selectitem(head,item,"single")'*(1:rows(item))
    endif

; Model specification-------------------------

    yy=y
    i=1
    while (i<=cols(y))
      if (header.natlog[i])
        yy[,i]=log(y[,i])
      else
        yy[,i]=y[,i]
      endif
      i=i+1
    endo
    if ((nocancel)&&(task=="specification"))
      switch
      case (type==1)
        modelfr(type,yt',header,outp,yy')  
        break
      endsw 
    endif 

; Output filename-------------------------------
    if ((nocancel)&&(task=="output"))    
      filename=readvalue("Output filename",filename)
      if (filename == "")
        outp = 0
      else
        outp = 1
        output(filename, "reset") 
      endif
    endif
; end of program loop-----------------------------
  endo
  if (outp==1)
    output(filename,"close")
  endif
endp
                              

