% function kalman_ipol_bgw_aly.m interpolates an annual time series (e.g. real GDP) using
% a slightly modified version of the interpolation procedure proposed by 
% Bernanke, Gertler, and Watson (1997): "Systematic Monetary Policy and the Effects of Oil Price Shocks", 
% Brookings Papers on Economic Activity, 1997(1), 91-157.
% copyright : Emanuel Mnch, Humboldt University Berlin, June 2003, 
% Email: moench@wiwi.hu-berlin.de  
% 
%*****************************************************************************************************
%**                                                                                                 **
%**   Inputs :     ser_aly : the annual time series to be interpolated                              **
%**                    ins : the matrix of monthly time series to be used as related variables      **
%**                          !! Make sure all series start in the same period !!                    **
%**                                                                                                 **
%**  Outputs :    ser_ipol : Kalman filter estimate of the interpolated monthly series,             **
%**                          i.e. the observation in t is generated using only information up to t  **
%**      ser_ipol_smoothed : Kalman smoother estimate based on the                                  **
%**                         full information set, i.e. the observation in t                         **
%**                         is generated using also information ahead of t                          **
%**                                                                                                 **
%*****************************************************************************************************
% Other functions needed : kalman_filter_bgw_aly.m, kalman_smoother_bgw_aly.m,
% loglik_bgw_aly.m, ipol_diag_bgw.m, nantozero.m


function ser_ipol = kalman_ipol_bgw3_aly(ser_aly, instruments);

y_data_aly = find(nantozero(ser_aly));
[r_ins, c_ins] = size(instruments);
length_x_data = zeros(c_ins,1);
for i = 1:c_ins,
    length_x_data(i) = max(find(nantozero(instruments(:,i))));
end;
length_ins = min(length_x_data);

first_obs_aly = 12*(y_data_aly(1,1)-1) + 1;
last_obs_aly = min([12*y_data_aly(length(y_data_aly)) length_ins]);
y_bar_aly = ser_aly;
y_plus = zeros(last_obs_aly,1);
pickdate_aly = 12:12:last_obs_aly;
y_plus(pickdate_aly,1) = y_bar_aly(1:floor(last_obs_aly/12));

last_obs = last_obs_aly;
timetrend = (1:1:last_obs)';
x = [ones(last_obs,1) instruments(1:last_obs,:)];
[length_ins num_ins] = size(x);

x_a = zeros(length(y_data_aly),num_ins);
for i = (first_obs_aly + 12- mod(first_obs_aly,12)) : 12 : 12*y_data_aly(length(y_data_aly)),
    x_a(i/12-y_data_aly(1)+1,:) = mean(x(i-11:i,:));
end;


% Initialize parameter vector
beta = inv(x_a'*x_a)*x_a'*y_bar_aly(y_data_aly); 
e = y_bar_aly(y_data_aly) - x_a*beta;
sig_e = sqrt(e'*e/length(y_bar_aly(y_data_aly) - num_ins));
r_squared = 1-(sig_e*sig_e/(std(y_bar_aly(y_data_aly))*std(y_bar_aly(y_data_aly))));
disp('R-squared of regression to determine initial parameter values :   '), disp(r_squared);
e1 = e(1:length(e)-1);
e2 = e(2:length(e));
rho = e2'*e1/(e2'*e2);
param_in = [beta' rho sig_e]';


% Find optimal parameter estimates
options = optimset('Display','iter','TolFun',1e-8,'maxfunevals',2000);
[param_out, fval] = fminsearch(@loglik_bgw_aly,param_in,options,x,y_plus);
disp('Parameter estimates: '); disp(param_out);
disp('Loglikelihood:  '); disp(fval);


y_ipol = [];
% Compute state vector using parameter estimates
    [xi_TT_out,P_TT_out,xi_TL_out,P_TL_out,nu_out,v_out] = kalman_filter_bgw_aly(param_out,x,y_plus);
    for i = 1: length(xi_TT_out);
        y_ipol = [y_ipol' xi_TT_out(1,1,i)']';
    end;
ser_ipol = y_ipol;


y_ipol_smoothed = [];
[xi_smoothed,P_smoothed] = kalman_smoother_bgw3_qlyaly(param_out,xi_TT_out,P_TT_out,xi_TL_out,P_TL_out);
for i = 1: length(xi_smoothed);
         y_ipol_smoothed = [y_ipol_smoothed' xi_smoothed(1,1,i)']';
end;
y_ipol_smoothed = flipdim(y_ipol_smoothed,1);
ser_ipol_smoothed = y_ipol_smoothed;
ser_ipol = ser_ipol_smoothed;

[r2_filtered, r2_smoothed] = ipol_diag_bgw(ser_ipol(first_obs_aly:last_obs_aly),ser_ipol_smoothed(first_obs_aly:last_obs_aly),P_TT_out,P_smoothed);

disp('Accuracy of interpolation ');
disp('R-squared y_ipol :             '), disp(r2_filtered);
disp('R-squared y_ipol_smoothed :    '), disp(r2_smoothed);




