% function kalman_ipol_bgw.m; interpolates a quarterly time series (e.g. real GDP) using
% a slightly modified version of the interpolation procedure proposed by 
% Bernanke, Gertler, and Watson (1997): "Systematic Monetary Policy and the Effects of Oil Price Shocks", 
% Brookings Papers on Economic Activity, 1997(1), 91-157.
% copyright : Emanuel Mnch, Humboldt University Berlin, June 2003, 
% Email: moench@wiwi.hu-berlin.de  
% 
%**************************************************************************
%
% Inputs :     ser : the quarterly time series to be interpolated 
%              ins : the matrix of monthly time series to be used as related variables
%              !! Make sure all series start in the same period !!
%
% Outputs :    ser_ipol : Kalman filter estimate of the interpolated
%                         monthly series.i.e. the observation in t is generated 
%                         using only information up to t
%     ser_ipol_smoothed : Kalman smoother estimate based on the
%                         full information set, i.e. the observation in t 
%                         is generated using also information ahead of t
% 
%***********************************************************************
% Other functions needed : kalman_filter_bgw.m, kalman_smoother_bgw.m,
% loglik_bgw.m, ipol_diag_bgw.m, nantozero.m

function [ser_ipol, ser_ipol_smoothed] = kalman_ipol_bgw(ser, ins);

y_bar1 = ser;
y_data = find(nantozero(y_bar1));
[num_obs num_ins] = size(ins);
min_obs_ins = length(ins);
for j = 1 : num_ins,
    num_obs_j = max(find(nantozero(ins(:,j))));
    if num_obs_j < min_obs_ins, min_obs_ins = num_obs_j; end;
end;
y_bar2 = y_bar1(y_data(1):y_data(length(y_data)));
first_obs = 3*(y_data(1,1)-1) + 1;
last_obs = min([3*y_data(length(y_data)) min_obs_ins]);
last_obs_qly = floor(last_obs/3);
y_bar = ser;
y_plus = zeros(last_obs,1);
pickdate = 3:3:last_obs;
y_plus(pickdate) = y_bar(1:y_data(last_obs_qly));

x = [ones(last_obs,1) ins(1:last_obs,:)];
[length_ins num_ins] = size(x);

x_q = zeros(length(y_data),num_ins);
for i = (first_obs + 3- mod(first_obs,3)) : 3 : 3*y_data(last_obs_qly),
    x_q(i/3-y_data(1)+1,:) = mean(x(i-2:i,:));
end;


% Initialize parameter vector
beta = inv(x_q'*x_q)*x_q'*y_bar2; 
e = y_bar2 - x_q*beta;
sig_e = sqrt(e'*e/length(y_bar2 - num_ins));
% r_squared = 1-(sig_e*sig_e/(std(y_bar2)*std(y_bar2)));
% disp('R-squared of regression to determine initial parameter values :   '), disp(r_squared);
e1 = e(1:length(e)-1);
e2 = e(2:length(e));
rho = e2'*e1/(e2'*e2);
param_in = [beta' rho sig_e]';


% Find optimal parameter estimates
options = optimset('Display','iter','TolFun',1e-6,'maxfunevals',2000);
[param_out, fval] = fminsearch(@loglik_bgw,param_in,options,x,y_plus);
disp('Parameter estimates: '); disp(param_out);
disp('Loglikelihood:  '); disp(fval);

y_ipol = [];
% Compute state vector using parameter estimates
    [xi_TT_out,P_TT_out,xi_TL_out,P_TL_out,nu_out,v_out] = kalman_filter_bgw(param_out,x,y_plus);
    for i = 1: length(xi_TT_out);
        y_ipol = [y_ipol' xi_TT_out(1,1,i)']';
    end;
save P_TL_out; save P_TT_out; save nu_out; save v_out;
ser_ipol = y_ipol;


y_ipol_smoothed = [];
[xi_smoothed,P_smoothed] = kalman_smoother_bgw(param_out,xi_TT_out,P_TT_out,xi_TL_out,P_TL_out);
for i = 1 : length(xi_smoothed);
    y_ipol_smoothed = [y_ipol_smoothed' xi_smoothed(1,1,i)']';
end;
y_ipol_smoothed = flipdim(y_ipol_smoothed,1);
ser_ipol_smoothed = y_ipol_smoothed;

[r2_filtered, r2_smoothed] = ipol_diag_bgw(ser_ipol,ser_ipol_smoothed,P_TT_out,P_smoothed);

disp('Accuracy of interpolation ');
disp('R-squared y_ipol :             '), disp(r2_filtered);
disp('R-squared y_ipol_smoothed :    '), disp(r2_smoothed);







