% function kalman_filter_bgw.m performs the Kalman filter iterations given the
% parameters to be estimated (param_vec) and the input data (x and y_bar)
% The output are three-dimensional arrays for the state vector (xi_out), 
% its variance-covariance matrix (P_out), the prediction error (nu_out), 
% and its variance-covariance matrix (v_out), repectively  
% copyright : Emanuel Mnch, Humboldt University Berlin, June 2003, 
% Email: moench@wiwi.hu-berlin.de  

function [xi_TT_out,P_TT_out,xi_TL_out,P_TL_out,nu_out,v_out] = kalman_filter_bgw(param_vec,x,y_bar);

beta = param_vec(1:length(param_vec)-2);
rho = param_vec(length(param_vec)-1);
sig_e = param_vec(length(param_vec))*param_vec(length(param_vec));

kappa = 100000;
T = length(y_bar);

% Initialize coefficient matrices
F = vertcat([0 0 0 rho],[1 0 0 0],[0 1 0 0],[0 0 0 rho]);
G = vertcat([1 0 0 0],[0 0 0 0],[0 0 0 0],[1 0 0 0]);
Q = sig_e*eye(4,4);

% Initialization of state vector and its variance-covariance matrix
xi_0 = zeros(4,1);
P_0 = kappa*eye(4,4);


% Kalman filter recursions
t = 1;
xi_LL = xi_0;
P_LL = P_0;
xi_TT_out = [];
xi_TL_out = [];
P_TT_out = [];
P_TL_out = [];
nu_out = [];
v_out = [];

while t <= T,
    x_t = x(t,:);
    yt_bar = y_bar(t,:);
    z_t = zeros(4,1);
    z_t(1) = x_t*beta;

    % Forecasting equations
    xi_TL = F*xi_LL + z_t;
    xi_TL_out = cat(3,xi_TL_out,xi_TL);
    P_TL = F*P_LL*F' + G*Q*G';
    P_TL_out = cat(3,P_TL_out,P_TL);
    
    % Updating equations
    if ((t >= min(find(y_bar))) & (mod(t,3) == 0)), 
        h_t = 1/3*([1 1 1 0])';
        %h_t = ([0 1 0 0])';
        K_t = P_TL*h_t*inv(h_t'*P_TL*h_t);
    else,
        h_t = zeros(4,1);
        K_t = zeros(4,1);
    end;
    xi_TT = xi_TL + K_t*(yt_bar - h_t'*xi_TL);
    xi_TT_out = cat(3,xi_TT_out,xi_TT);
    P_TT = P_TL - K_t*h_t'*P_TL;
    P_TT_out = cat(3,P_TT_out,P_TT);
        
    % Prediction error decomposition
    nu_t = yt_bar - h_t'*xi_TL;
    nu_out = cat(3,nu_out,nu_t);
    v_t = h_t'*P_TL*h_t;
    v_out = cat(3,v_out,v_t);
    
    xi_LL = xi_TT;
    P_LL = P_TT;
    
    t = t + 1;
end;

